/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef MOZILLA_LAYERS_TEXTUREFORWARDER
#define MOZILLA_LAYERS_TEXTUREFORWARDER

#include <stdint.h>  // for int32_t, uint64_t
#include "gfxTypes.h"
#include "mozilla/layers/LayersMessages.h"  // for Edit, etc
#include "mozilla/layers/LayersTypes.h"     // for LayersBackend
#include "mozilla/layers/TextureClient.h"   // for TextureClient
#include "mozilla/layers/KnowsCompositor.h"
#include "nsISerialEventTarget.h"

namespace mozilla {
namespace ipc {
class IShmemAllocator;
}
namespace layers {
class CanvasChild;

/**
 * An abstract interface for classes that implement the autogenerated
 * IPDL actor class. Lets us check if they are still valid for IPC.
 */
class LayersIPCActor {
 public:
  virtual bool IPCOpen() const { return true; }
};

/**
 * An abstract interface for LayersIPCActors that implement a top-level
 * IPDL protocol so also have their own channel.
 * Has their own MessageLoop for message dispatch, and can allocate
 * shmem.
 */
class LayersIPCChannel : public LayersIPCActor,
                         public mozilla::ipc::IShmemAllocator {
 public:
  NS_INLINE_DECL_PURE_VIRTUAL_REFCOUNTING

  virtual bool IsSameProcess() const = 0;

  virtual bool UsesImageBridge() const { return false; }

  virtual base::ProcessId GetParentPid() const = 0;

  virtual nsISerialEventTarget* GetThread() const = 0;

  virtual FixedSizeSmallShmemSectionAllocator* GetTileLockAllocator() {
    return nullptr;
  }

  virtual void CancelWaitForNotifyNotUsed(uint64_t aTextureId) = 0;

  virtual wr::MaybeExternalImageId GetNextExternalImageId() {
    return Nothing();
  }

 protected:
  virtual ~LayersIPCChannel() = default;
};

/**
 * An abstract interface for classes that can allocate PTexture objects
 * across IPDL. Currently a sub-class of LayersIPCChannel for simplicity
 * since all our implementations use both, but could be independant if needed.
 */
class TextureForwarder : public LayersIPCChannel {
 public:
  /**
   * Create a TextureChild/Parent pair as as well as the TextureHost on the
   * parent side.
   */
  virtual PTextureChild* CreateTexture(
      const SurfaceDescriptor& aSharedData, const ReadLockDescriptor& aReadLock,
      LayersBackend aLayersBackend, TextureFlags aFlags, uint64_t aSerial,
      wr::MaybeExternalImageId& aExternalImageId,
      nsISerialEventTarget* aTarget = nullptr) = 0;

  /**
   * Returns the CanvasChild for this TextureForwarder.
   */
  virtual already_AddRefed<CanvasChild> GetCanvasChild() { return nullptr; };
};

}  // namespace layers
}  // namespace mozilla

#endif
