import buildconfig
import six
import yaml
from mozbuild.preprocessor import Preprocessor

HEADER_TEMPLATE = """\
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef %(includeguard)s
#define %(includeguard)s

/* This file is generated by wasm/GenerateBuiltinModules.py. Do not edit! */

%(contents)s

#endif // %(includeguard)s
"""


def generate_header(c_out, includeguard, contents):
    c_out.write(
        HEADER_TEMPLATE
        % {
            "includeguard": includeguard,
            "contents": contents,
        }
    )


def load_yaml(yaml_path):
    # First invoke preprocessor.py so that we can use #ifdef JS_SIMULATOR in
    # the YAML file.
    pp = Preprocessor()
    pp.context.update(buildconfig.defines["ALLDEFINES"])
    pp.out = six.StringIO()
    pp.do_filter("substitution")
    pp.do_include(yaml_path)
    contents = pp.out.getvalue()
    return yaml.safe_load(contents)


def cppBool(v):
    if v:
        return "true"
    return "false"


def main(c_out, yaml_path):
    data = load_yaml(yaml_path)

    # Iterate for all defined builtin methods
    contents = "#define FOR_EACH_BUILTIN_MODULE_FUNC(M) \\\n"
    for i in range(len(data)):
        op = data[i]
        sa = op["symbolic_address"]
        contents += (
            f"    M({op['op']}, \"{op['export']}\", "
            f"{sa['name']}, {sa['type']}, {op['entry']}, {cppBool(op['uses_memory'])}, {i})\\\n"
        )
    contents += "\n"

    for op in data:
        # Define DECLARE_BUILTIN_MODULE_FUNC_PARAM_VALTYPES_<op> as:
        # `{ValType::I32, ValType::I32, ...}`.
        contents += (
            f"#define DECLARE_BUILTIN_MODULE_FUNC_PARAM_VALTYPES_{op['op']} "
            f"{{{', '.join(op['params'])}}}\n"
        )

        # Define DECLARE_BUILTIN_MODULE_FUNC_PARAM_SASTYPES_<op> as:
        # `<num_types>, {_PTR, _I32, ..., _PTR, _END}`.
        num_types = len(op["params"]) + 1
        sas_types = (
            f"{{_PTR{''.join(', ' + (p + '.toMIRType()') for p in op['params'])}"
        )
        if op["uses_memory"]:
            sas_types += ", _PTR"
            num_types += 1
        sas_types += ", _END}"

        contents += f"#define DECLARE_BUILTIN_MODULE_FUNC_PARAM_SASTYPES_{op['op']} {num_types}, {sas_types}\n"

        result_valtype = ""
        result_sastype = ""
        if "result" in op:
            result_valtype = f"Some({op['result']})\n"
            result_sastype = f"{op['result']}.toMIRType()\n"
        else:
            result_valtype = "Nothing()"
            result_sastype = "_VOID"
        contents += f"#define DECLARE_BUILTIN_MODULE_FUNC_RESULT_VALTYPE_{op['op']} {result_valtype}\n"
        contents += f"#define DECLARE_BUILTIN_MODULE_FUNC_RESULT_SASTYPE_{op['op']} {result_sastype}\n"
        contents += f"#define DECLARE_BUILTIN_MODULE_FUNC_FAILMODE_{op['op']} _{op['fail_mode']}\n"

    generate_header(c_out, "wasm_WasmBuiltinModuleGenerated_h", contents)
