/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Mozilla SMIL module.
 *
 * The Initial Developer of the Original Code is Brian Birtles.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Brian Birtles <birtles@gmail.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#ifndef NS_SMILINTERVAL_H_
#define NS_SMILINTERVAL_H_

#include "nsSMILInstanceTime.h"
#include "nsTArray.h"

//----------------------------------------------------------------------
// nsSMILInterval class
//
// A structure consisting of a begin and end time. The begin time must be
// resolved (i.e. not indefinite or unresolved).
//
// For an overview of how this class is related to other SMIL time classes see
// the documentation in nsSMILTimeValue.h

class nsSMILInterval
{
public:
  nsSMILInterval();
  nsSMILInterval(const nsSMILInterval& aOther);
  ~nsSMILInterval();
  void NotifyChanged(const nsSMILTimeContainer* aContainer);
  void NotifyDeleting();

  const nsSMILInstanceTime* Begin() const
  {
    NS_ABORT_IF_FALSE(mBegin && mEnd,
        "Requesting Begin() on un-initialized instance time");
    return mBegin;
  }
  nsSMILInstanceTime* Begin();

  const nsSMILInstanceTime* End() const
  {
    NS_ABORT_IF_FALSE(mBegin && mEnd,
        "Requesting End() on un-initialized instance time");
    return mEnd;
  }
  nsSMILInstanceTime* End();

  void SetBegin(nsSMILInstanceTime& aBegin);
  void SetEnd(nsSMILInstanceTime& aEnd);
  void Set(nsSMILInstanceTime& aBegin, nsSMILInstanceTime& aEnd)
  {
    SetBegin(aBegin);
    SetEnd(aEnd);
  }

  void FreezeBegin()
  {
    NS_ABORT_IF_FALSE(mBegin && mEnd,
        "Freezing Begin() on un-initialized instance time");
    mBegin->MarkNoLongerUpdating();
  }

  void FreezeEnd()
  {
    NS_ABORT_IF_FALSE(mBegin && mEnd,
        "Freezing End() on un-initialized instance time");
    NS_ABORT_IF_FALSE(!mBegin->MayUpdate(),
        "Freezing the end of an interval without a fixed begin");
    mEnd->MarkNoLongerUpdating();
  }

  // XXX Backwards seeking support (bug 492458)
  void Unfreeze()
  {
    // XXX
    UnfreezeEnd();
  }

  void UnfreezeEnd()
  {
    // XXX
  }

  void AddDependentTime(nsSMILInstanceTime& aTime);
  void RemoveDependentTime(const nsSMILInstanceTime& aTime);

private:
  nsRefPtr<nsSMILInstanceTime> mBegin;
  nsRefPtr<nsSMILInstanceTime> mEnd;

  typedef nsTArray<nsRefPtr<nsSMILInstanceTime> > InstanceTimeList;
  InstanceTimeList mDependentTimes;

  // When change notifications are passed around the timing model we try to
  // filter out all changes where there is no observable difference to an
  // instance time. Changes that may produce an observable difference are:
  //
  // * Changes to the time of an interval endpoint
  // * Changes in the relative times of different time containers
  // * Changes to the dependency chain (which may affect the animation sandwich)
  //
  // The nsSMILTimeValueSpec can detect the first two changes by recalculating
  // the time but in order to help detect the third change we simply set a flag
  // whenever the mBegin or mEnd pointers are changed. These flags are reset
  // when the next change notification is sent.
  PRPackedBool mBeginObjectChanged;
  PRPackedBool mEndObjectChanged;
};

#endif // NS_SMILINTERVAL_H_
